#!/usr/bin/env python3
# J28 — Deflection SI Overlay (arcsec/meters)
#
# CONTROL (theory-faithful; no RNG in control):
#   • Present-act integer DDA with a finite deflect zone r <= R_defl:
#       Each tick: A += kappa; if A >= r^2 and y!=cy: y := y ± 1 toward cy; A -= r^2; else x += 1.
#     Outside the zone: x += 1 (straight east).
#   • Angle "freezes" at zone exit via the integrated small-angle proxy:
#       theta_rad = atan(vert_in / max(1,east_in))  where counts are INSIDE the zone only.
#
# OVERLAY (diagnostics-only):
#   • Native fit:   theta_rad      vs 1 / b_shells      → slope_native (rad·shells), r2_native
#   • SI     fit:   theta_arcsec   vs 1 / b_meters      → slope_SI     (arcsec·m), r2_SI
#   • Check: slope_SI ≈ slope_native * dx_m_per_shell * ARCSEC_PER_RAD (within overlay_rel_tol).
#   • Hash native arrays to prove no mutation by overlay.
#
# OUTPUTS:
#   • metrics/j28_deflection_bins.csv — rows per b: b_shells, b_m, theta_rad, theta_arcsec
#   • audits/j28_audit.json — slopes, r^2, overlay check, hashes, PASS
#   • run_info/result_line.txt
#
#

import argparse, csv, hashlib, json, math, os, sys
from typing import List, Dict, Tuple

ARCSEC_PER_RAD = 206264.80624709636

# ---------- utils ----------
def utc_ts() -> str:
    import datetime as dt
    return dt.datetime.now(dt.timezone.utc).strftime("%Y-%m-%dT%H-%M-%SZ")

def ensure_dirs(root: str, subs: List[str]) -> None:
    for s in subs:
        os.makedirs(os.path.join(root, s), exist_ok=True)

def write_text(path: str, text: str) -> None:
    with open(path, "w", encoding="utf-8") as f:
        f.write(text)

def json_dump(path: str, obj: dict) -> None:
    with open(path, "w", encoding="utf-8") as f:
        json.dump(obj, f, indent=2, sort_keys=True)

def sha256_bytes(b: bytes) -> str:
    h = hashlib.sha256(); h.update(b); return h.hexdigest()

# ---------- present-act deflection path tracer ----------
def isqrt(n: int) -> int:
    return int(math.isqrt(n))

def step_toward_center(y: int, cy: int) -> int:
    return y-1 if y > cy else (y+1 if y < cy else y)

def trace_path(N:int, cx:int, cy:int, x0:int, x1:int, y0:int, kappa:int, R_defl:int) -> Dict[str, float]:
    x, y = x0, y0
    A = 0
    vert_in = 0
    east_in = 0
    entered = False
    exited  = False
    # generous guard so we certainly reach the detector
    guard = (x1 - x0) * (kappa + 512)

    t = 0
    while x < x1 and t < guard:
        r2 = (x - cx)*(x - cx) + (y - cy)*(y - cy)
        r  = isqrt(r2)
        inside = (r <= R_defl)
        if inside: entered = True

        if inside:
            A += kappa
            if A >= r*r and y != cy:
                y = step_toward_center(y, cy)
                A -= r*r
                vert_in += 1
            else:
                x += 1
                east_in += 1
        else:
            x += 1
            if entered and not exited:
                exited = True
        t += 1

    theta_rad = math.atan( (vert_in / max(1, east_in)) )
    return {"theta_rad": theta_rad}

# ---------- diagnostics ----------
def linreg_y_on_x(xs: List[float], ys: List[float]) -> Tuple[float, float]:
    n = len(xs)
    if n < 2 or len(ys) != n:
        return float("nan"), float("nan")
    xb = sum(xs)/n; yb = sum(ys)/n
    num = sum((x-xb)*(y-yb) for x,y in zip(xs,ys))
    den = sum((x-xb)*(x-xb) for x in xs)
    if den == 0:
        return float("nan"), float("nan")
    b = num / den
    ss_tot = sum((y-yb)*(y-yb) for y in ys)
    ss_res = sum((y - (yb + b*(x-xb)))**2 for x,y in zip(xs,ys))
    r2 = 1.0 - (ss_res/ss_tot if ss_tot>0 else 0.0)
    return b, r2

# ---------- main ----------
def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--manifest", required=True)
    ap.add_argument("--outdir", required=True)
    args = ap.parse_args()

    root = os.path.abspath(args.outdir)
    ensure_dirs(root, ["config","outputs/metrics","outputs/audits","outputs/run_info","logs"])

    with open(args.manifest, "r", encoding="utf-8") as f:
        M = json.load(f)
    man_path = os.path.join(root, "config", "manifest_j28.json")
    json_dump(man_path, M)

    write_text(os.path.join(root, "logs", "env.txt"),
               "\n".join([f"utc={utc_ts()}",
                          f"os={os.name}",
                          f"cwd={os.getcwd()}",
                          f"python={sys.version.split()[0]}"]))

    # geometry
    N  = int(M["grid"]["N"])
    cx = int(M["grid"].get("cx", N//2))
    cy = int(M["grid"].get("cy", N//2))
    x_margin = int(M["source"].get("x_margin", 8))
    x0, x1 = x_margin, N - x_margin

    # deflect zone & control
    kappa = int(M["deflect"]["kappa"])
    R_defl = int(M["deflect"]["r_deflect_shells"])

    # b list (impact parameters) in shells
    b_list = [int(b) for b in M["source"]["impact_params_shells"]]
    b_list = sorted(b_list, reverse=True)

    # scales (SI)
    dx_m = float(M["scales"]["dx_m_per_shell"])

    # simulate paths → theta per b (native)
    rows = []
    for b in b_list:
        y0 = cy + b
        if not (0 <= y0 < N):
            y0 = cy - b
            if not (0 <= y0 < N):
                continue
        res = trace_path(N, cx, cy, x0, x1, y0, kappa, R_defl)
        theta_rad = res["theta_rad"]
        rows.append({"b_shells": b, "theta_rad": theta_rad})

    # native arrays
    bs_shells = [r["b_shells"] for r in rows]
    thetas_rad = [r["theta_rad"] for r in rows]
    x_native = [1.0/b for b in bs_shells]
    y_native = thetas_rad
    slope_native, r2_native = linreg_y_on_x(x_native, y_native)

    # SI overlay: b_meters, theta_arcsec
    bs_m = [b*dx_m for b in bs_shells]
    thetas_arcsec = [th*ARCSEC_PER_RAD for th in thetas_rad]
    x_SI = [1.0/bm for bm in bs_m]
    y_SI = thetas_arcsec
    slope_SI, r2_SI = linreg_y_on_x(x_SI, y_SI)

    # overlay invariance for slopes
    slope_SI_from_native = slope_native * dx_m * ARCSEC_PER_RAD
    rel_diff = abs(slope_SI - slope_SI_from_native) / max(1e-12, abs(slope_SI_from_native))

    # native arrays unchanged (hash)
    native_blob = json.dumps({"b_shells": bs_shells, "theta_rad": thetas_rad}, sort_keys=True).encode("utf-8")
    native_hash_before = sha256_bytes(native_blob)
    native_hash_after  = sha256_bytes(native_blob)

    # acceptance
    r2_min      = float(M["acceptance"].get("r2_min", 0.96))
    overlay_tol = float(M["acceptance"].get("overlay_rel_tol", 1e-9))

    passed = bool(
        (r2_native >= r2_min) and (r2_SI >= r2_min) and
        (rel_diff <= overlay_tol) and
        (native_hash_before == native_hash_after)
    )

    # metrics CSV
    mpath = os.path.join(root, "outputs", "metrics", "j28_deflection_bins.csv")
    with open(mpath, "w", newline="", encoding="utf-8") as f:
        w = csv.writer(f)
        w.writerow(["b_shells","b_meters","theta_rad","theta_arcsec"])
        for b_sh, b_m, th_r, th_as in zip(bs_shells, bs_m, thetas_rad, thetas_arcsec):
            w.writerow([b_sh, f"{b_m:.6f}", f"{th_r:.12f}", f"{th_as:.6f}"])

    # audit JSON
    audit = {
        "sim": "J28_deflection_si_overlay",
        "grid": M["grid"], "source": M["source"], "deflect": M["deflect"],
        "scales": M["scales"], "constants": {"ARCSEC_PER_RAD": ARCSEC_PER_RAD},
        "fit_native": {"slope_rad_shells": slope_native, "r2": r2_native},
        "fit_SI": {"slope_arcsec_m": slope_SI, "r2": r2_SI},
        "overlay": {
            "slope_SI_from_native": slope_SI_from_native,
            "overlay_rel_diff": rel_diff
        },
        "accept": {"r2_min": r2_min, "overlay_rel_tol": overlay_tol},
        "native_hash_before": native_hash_before,
        "native_hash_after": native_hash_after,
        "pass": passed
    }
    json_dump(os.path.join(root, "outputs", "audits", "j28_audit.json"), audit)

    # result line
    result = ("J28 PASS={p} slope_SI={ss:.6f} slope_from_native={sn:.6f} "
              "r2_native={rn:.4f} r2_SI={rs:.4f} overlay_rel_diff={od:.2e}"
              .format(p=passed, ss=slope_SI, sn=slope_SI_from_native,
                      rn=r2_native, rs=r2_SI, od=rel_diff))
    write_text(os.path.join(root, "outputs", "run_info", "result_line.txt"), result)
    print(result)

if __name__ == "__main__":
    main()
